!
! =====================================================================
!
! File: rng.f90
!
! Fortran 90 source code for parallel implementation of mzran
! random number generator.
!
! Also required: file mzranseed.dat.
!
! Hannover, Germany, September 2012
!
! Manfred Mudelsee
! Climate Risk Analysis
! http://www.climate-risk-analysis.com
!
! =====================================================================
!
module rng
      use nrtype
      implicit none
      private
      public :: factor,mzran,rng_t,rng_initial_seed,rng_seed,ns1
      integer, parameter :: ns1=100000 ! 1st dimension of the state
!                                      !    ns1 should be .ge. the
!                                      !    maximum number of threads
!                                      !    e.g., if a do loop j = 1, b
!                                      !    is to be parallelized,
!                                      !    then ns1 .ge. b
!                                      !    is required
      integer, parameter :: ns2=4      ! 2nd dimension of the state
      integer, dimension(1:ns1,1:ns2) :: &
         default_seed ! default seed vector
      real(sp), parameter :: factor=2.328306435454e-10_sp
!        this constant is equal to 0.5 / 2,147,483,649
      type :: rng_t
         integer, dimension(ns2) :: state
      end type rng_t
contains
subroutine rng_initial_seed
      implicit none
!     Initializes default_seed.
      character(len=1) :: header
      character(len=*), parameter :: seedfile= &
         'mzranseed.dat'            ! put this file into
                                    ! same directory the executable
      integer :: i
      integer :: j
      integer :: openerror
      open (unit=1, file=seedfile, status='old', form='formatted', &
         action='read', iostat=openerror)
      if (openerror /= 0 ) then
         print *,'Error during seed file opening:',seedfile
         return
      end if
      read (unit=1, fmt='(a1)') header
      do i=1,ns1
         read (unit=1, fmt='(4(1x,i11))') default_seed(i,1:4)
      end do
      close (unit=1, status='keep')
end subroutine rng_initial_seed
subroutine rng_seed(self,iseed)
      implicit none
      type(rng_t), intent(inout) :: self
      integer, intent(in) :: iseed
!     Seeds mzran to default_seed using a single integer, iseed.
      self%state(1:ns2)=default_seed(iseed,1:ns2)
end subroutine rng_seed
function mzran(self)
      use nrtype
      implicit none
      type(rng_t), intent(inout) :: self
      real(sp) :: mzran
!
!     Draws a random real number uniformly distributed over [0; 1],
!     exclusive of the endpoints.
!     
!     References:
!
!     Blevins J (2009) Parallel Computing in Fortran with OpenMP.
!        http://jblevins.org/log/openmp
!        (access date: 14 August 2012).
!        [hints for parallelization]
!
!     Marsaglia G, Zaman A (1994) Some portable very-long-period
!        random number generators. Computers in Physics 8:117-121.
!        [source code mzran]
!
      integer :: imz
      imz=self%state(1)-self%state(3)
      if (imz < 0) imz=imz+2147483579
      self%state(1)=self%state(2)
      self%state(2)=self%state(3)
      self%state(3)=imz
      self%state(4)=69069*self%state(4)+1013904243
      imz=imz+self%state(4)
      mzran=0.5_sp+factor*imz
end function mzran
end module rng
!
! =====================================================================
!
