!
! LINCAL:
!
! A software for calibrating linear climate proxy models
!
! ==================================================================================================
!
! Change log
! ==========
!
! Version     Date                 Comments
!
! 1.01        December 2023        o bibliographical details added
! 1.00        December 2022        o original version
!
! ==================================================================================================
!
! Copyright notice:     Copyright (C) Manfred Mudelsee
!
!                       https://www.manfredmudelsee.com
!
! Permission notice:    Permission is hereby granted, free of charge, to any person
!                       obtaining a copy of this software and associated computer files,
!                       to deal in the Software without restriction, including without
!                       limitation the rights to use, copy, modify, merge, publish,
!                       distribute, sublicense, and/or sell copies of the Software, and to
!                       permit persons to whom the Software is furnished to do so, subject
!                       to the following conditions:
!
!                       The above copyright notice, including the URL, and this permission
!                       notice shall be included in all copies or substantial portions of
!                       the Software.
!
! ==================================================================================================
!
! Disclaimer:           THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
!                       EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
!                       MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
!                       NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
!                       BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
!                       ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
!                       CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
!                       SOFTWARE.
!
! ==================================================================================================
!
! Author
! ======
!
! Manfred Mudelsee
!
! Climate Risk Analysis
! Kreuzstrasse 27
! Heckenbeck
! 37581 Bad Gandersheim
! Germany
!
! and
!
! University of Potsdam
! Institute of Geosciences
! Karl-Liebknecht-Strasse 24-25
! 14476 Potsdam-Golm
! Germany
!
! https://www.manfredmudelsee.com
!
! ==================================================================================================
!
! Language and development system
! ===============================
!
! Tarox Workstation 745TQ, 2 CPU Intel Xeon E5-2620 @2.00 GHz, 64 GB RAM;
! Windows 7 Professional Service Pack 1;
! Fortran 90 (including free and adapted Numerical Recipes code);
! Intel(R) Visual Fortran Composer XE 2013
!
! ==================================================================================================
!
! Acknowledgements
! ================
!
! This work has been funded by the Deutsche Forschungsgemeinschaft
! (DFG, German Research Foundation), project number 468589022 (M.M.),
! within the SPP 2299, project number 441832482.
!
! ==================================================================================================
!
include 'nrtype.f90'    ! free Numerical Recipes file
include 'nr.f90'        ! free Numerical Recipes file
include 'nrutil.f90'    ! free Numerical Recipes file
!
include 'rng.f90'       ! random number generator, requires file 'mzranseed.dat' to
                        ! reside in the same directory as the executable
!
! ==================================================================================================
!
module parameters
      use nrtype
      implicit none
!     Constants.
      save
!
! 1.  Minimum number of points
!     ========================
!
      integer, parameter :: nmin = 10
!
! 2.  Number of bootstrap simulations
!     ===============================
!
      integer, parameter :: b = 2000
!
! 3.  Confidence level (1 - 2 * alpha)
!     ================================
!
      real(sp), parameter :: alpha = 0.025_sp
!     real(sp), parameter :: alpha = 0.050_sp
!
! 4.  Output file name
!     ================
!
      character(len = 10), parameter :: outputfile = 'LINCAL.dat'
!
! 5.  Prediction points
!     =================
!
      real(sp), parameter :: fp = 0.5_sp    ! factor determines prediction range
      integer, parameter  :: np = 2000      ! number of prediction points
!
! 6.  Miscelleanous
!     =============
!
      integer, parameter :: ntry = 5        ! user input: maximum number of errors
!
end module parameters
!
! ==================================================================================================
!
module chixyfit
      use nrtype
      use nrutil, only : nrerror
      implicit none
!     Auxiliary module for subroutine wlsxy.
      real(sp), dimension(:), pointer :: xxp, yyp, sxp, syp, wwp
      real(sp) :: aa, offs
contains
function chixy(bang)
      implicit none
      real(sp), intent(in) :: bang
      real(sp) :: chixy
      real(sp), parameter :: big = 1.0e30_sp
      real(sp) :: avex
      real(sp) :: avey
      real(sp) :: sumw
      real(sp) :: b
      if (.not. associated(wwp)) call nrerror("module chixy: bad pointers")
      b = tan(bang)
      wwp(:) = (b*sxp(:))**2 + syp(:)**2
      where (wwp(:) < 1.0 / big)
            wwp(:) = big
      elsewhere
            wwp(:) = 1.0_sp / wwp(:)
      end where
      sumw = sum(wwp)
      avex = dot_product(wwp, xxp) / sumw
      avey = dot_product(wwp, yyp) / sumw
      aa = avey - b * avex
      chixy = sum(wwp(:) * (yyp(:) - aa - b * xxp(:))**2) - offs
end function chixy
end module chixyfit
!
! ==================================================================================================
!
module data1
      use nrtype
      implicit none
!     Data file (original data): n rows of (t, x, y); residuals.
      save
      real(sp), allocatable, dimension(:) :: t              ! time
      real(sp), allocatable, dimension(:) :: x              ! x-data
      real(sp), allocatable, dimension(:) :: y              ! y-data
!
      real(sp), allocatable, dimension(:) :: res_x          ! x-residuals
      real(sp), allocatable, dimension(:) :: res_y          ! y-residuals
end module data1
!
! ==================================================================================================
!
module prior_knowledge
      use nrtype
      implicit none
!     Prior knowledge about data errors.
      save
      real(sp) ::    sx_const = -999.0_sp           ! standard deviation       (constant) of x(i)
      real(sp) ::    sy_const = -999.0_sp           ! standard deviation       (constant) of y(i)
      real(sp) :: delta_const = -999.0_sp           ! standard deviation ratio (constant)
      real(sp) ::  sx_predict = -999.0_sp           ! standard deviation       (constant) of x(i)
                                                    !    for prediction
      real(sp), allocatable, dimension(:) :: sx     ! standard deviation                  of x(i)
      real(sp), allocatable, dimension(:) :: sy     ! standard deviation                  of y(i)
      real(sp), allocatable, dimension(:) :: delta  ! standard deviation ratio
end module prior_knowledge
!
! ==================================================================================================
!
module result1
      use nrtype
      use parameters, only: np
      implicit none
!
!     Result.
!     Note: CI, confidence interval.
!
      save
      real(sp) :: beta0     = -999.0_sp                 ! intercept,  estimate
      real(sp) :: beta0_low = -999.0_sp                 ! intercept,  CI, lower bound
      real(sp) :: beta0_upp = -999.0_sp                 ! intercept,  CI, upper bound
!                                                                 
      real(sp) :: beta1     = -999.0_sp                 ! slope,      estimate
      real(sp) :: beta1_low = -999.0_sp                 ! slope,      CI, lower bound
      real(sp) :: beta1_upp = -999.0_sp                 ! slope,      CI, upper bound
!
      real(sp), dimension(1:np) :: xp     = -999.0_sp   ! predictor
      real(sp), dimension(1:np) :: yp     = -999.0_sp   ! prediction
      real(sp), dimension(1:np) :: yp_low = -999.0_sp   ! prediction, CI, lower bound
      real(sp), dimension(1:np) :: yp_upp = -999.0_sp   ! prediction, CI, upper bound
end module result1
!
! ==================================================================================================
!
module setting
      use nrtype
      implicit none
!     Setting.
      save
      character (len = 200) :: datafile     ! input data file name
      integer :: datacase = - 999           ! describes situation of input data and
!                                           !    prior knowledge (see subroutine chsett1)
      integer  :: n     = -999              ! sample size
      integer  :: l_mbb = -999              ! MBB block length
      real(sp) :: tau_x = -999.0_sp         ! persistence time, res_x
      real(sp) :: tau_y = -999.0_sp         ! persistence time, res_y
      real(sp) :: phi_inv_alpha = -999.0_sp ! percentage point of the standard Gaussian distribution
      real(sp) :: t_ave_alpha   = -999.0_sp ! percentage point of Student's t distribution
end module setting
!
! ==================================================================================================
!
program lincal1
      use rng
      use prior_knowledge
      implicit none
      character(len = 9), parameter :: lseedfile = 'lseed.dat'
      integer :: iocheck = -999
      type(rng_t), dimension(1), save :: rng3                   ! seed of random number generator
      type(rng_t), dimension(1), save :: rng4                   ! seed of random number generator
      integer :: iseed   = 1                                    ! dummy integer for random number generator
      integer :: liseed  = -999
      integer :: liseed2 = -999
!
! 1.  Preliminary
!     ===========
!
      call info1                ! Welcomes
      call clearscreen          ! Clears screen
!
! 2.  Random number generator
!     =======================
!
!     Note: for details on the random number generator and its implementation, see
!           the software HT (https://www.manfredmudelsee.com/textbook/HT.zip).
!
      call rng_initial_seed
      open (unit = 1, file = lseedfile, status = 'old', form = 'formatted',                     &
            action = 'readwrite', iostat  =  iocheck)
      if (iocheck .eq. 0 ) then
         read (unit = 1, fmt = *) liseed
         liseed2 = mod(liseed, ns1) + 1
         rewind(1)
         write (unit = 1, fmt = '(i6)') mod(liseed + 1, ns1) + 1
         close (unit = 1, status = 'keep')
      else
         ! if no seedfile exists, start a new one
         liseed = 1
         liseed2 = 2
         open (unit = 1, file = lseedfile, status = 'new', form = 'formatted',                  &
               action = 'write')
         write (unit = 1, fmt = '(i6)') liseed + 2
         close (unit = 1, status = 'keep')
      end if
      call rng_seed(rng3(iseed), liseed)
      call rng_seed(rng4(iseed), liseed2)
!
! 3.  Data
!     =====
!
      call chsett1              ! Changes setting: datacase
      call chsett2              ! Changes setting: datafile
      call chsett3              ! Changes setting: n
      call allocate1            ! Allocates   data, residuals
      call init1                ! Initializes data, residuals
      call read1                ! Reads       data
      call calc_inv_alpha       ! Calculates phi_inv_alpha
      call calc_inv_student     ! Calculates t_ave_alpha
!
! 4.  Prior knowledge
!     ===============
!
      call chprior1             ! Changes prior knowledge: data errors
      call chprior2             ! Changes prior knowledge: prediction data errors
!
! 5.  Estimation and prediction
!     =========================
!
      call estimation1          ! Estimates beta0, beta1
      call prediction1          ! Predicts yp
!
! 6.  Confidence intervals
!     ====================
!
      call confidence1(rng3(iseed), rng4(iseed))    ! Determines confidence intervals
!
! 7.  Output and exit
!     ===============
!
      call output1              ! Outputs result
      call deallocate1          ! Deallocates data
      call info2                ! Says goodbye
!
end program lincal1
!
! ==================================================================================================
!
subroutine allocate1
      use nrtype
      use prior_knowledge, only: sx, sy, delta
      use setting, only: datacase, n
      use data1, only: t, x, y, res_x, res_y
      implicit none
!     Allocates data, residuals.
      integer :: error = 0
      if (datacase .eq. 1) then
         allocate(x(n), y(n), stat = error)
      else if (datacase .eq. 2) then   
         allocate(t(n), x(n), y(n), stat = error)
      else if (datacase .eq. 3) then
         allocate(x(n), y(n), sx(n), sy(n), delta(n), stat = error)
      else if (datacase .eq. 4) then   
         allocate(t(n), x(n), y(n), sx(n), sy(n), delta(n), stat = error)
      end if
      if (error /= 0) then
         print '(a)',' Subroutine allocate1 (part 1): space requested not possible - LINCAL terminates.'
         stop
      end if
      allocate(res_x(n), res_y(n), stat = error)
      if (error /= 0) then
         print '(a)',' Subroutine allocate1 (part 2): space requested not possible - LINCAL terminates.'
         stop
      end if      
end subroutine allocate1
!
! ==================================================================================================
!
subroutine avevar_n(x, n, ave, var)
      use nrtype
      implicit none
	  integer, intent(in) :: n                  ! sample size
      real(sp), dimension(n), intent(in) :: x   ! data
      real(sp), intent(out) :: ave              ! average
      real(sp), intent(out) :: var              ! variance
!
! 	  Calculates average and variance (the latter one with n instead of n - 1).
!
      real(sp), dimension(n) :: s
      ave = sum(x(:)) / n
      s(:) = x(:) - ave
      var = dot_product(s, s)
	  var = var / n
end subroutine avevar_n
!
! ==================================================================================================
!    
subroutine bootstrap1(n, x, y, l, x_resample, y_resample, rng3, rng4)
      use nrtype
      use rng
      implicit none
      integer, intent(in) :: n
      real(sp), dimension(n), intent(in)  :: x
      real(sp), dimension(n), intent(in)  :: y
      integer, intent(in) :: l
      real(sp), dimension(n), intent(out) :: x_resample
      real(sp), dimension(n), intent(out) :: y_resample
      type(rng_t), dimension(1), intent(inout) :: rng3          ! seeded random number generator
      type(rng_t), dimension(1), intent(inout) :: rng4          ! seeded random number generator
!
!     Pairwise MBB.
!
      integer, dimension(n) :: indxx
      integer :: i       = 0
      integer :: iseed   = 1                    ! dummy integer for random number generator
      integer :: j       = 0
      integer :: k       = 0
      integer :: idum    = 1
      integer :: n_block_start
      real(sp), dimension(1:n) :: w             ! random number over the interval [0; 1],
                                                ! exclusive of the endpoints
!
! 1.  Random numbers
!     ==============
!
      w(:) = -999.0_sp
      do i = 1, n
         w(i) = mzran(rng3(iseed))
      end do
!
! Test
!
!
!     open  (unit = 1, file = 'randomtest.dat', status = 'unknown', form = 'formatted', action = 'write')
!     do i = 1, 100000
!        write (unit = 1, fmt = '(3x, i7, 3x, f16.8)'), i, mzran(rng3(kran))
!     end do
!     close (unit = 1, status = 'keep')
!     print *, ' Random test dataset has been written'
!     read (*,*)
!
! 2.  Resampling
!     ==========
!
!     Note: if l = 1, we use ordinary bootstrap.
!
      n_block_start = n - l + 1
      if (l == 1) then
         do i = 1, n
            indxx(i) = int(n * w(i)) + 1
         end do
      else
         k = 1
out1:    do i = 1, n
            indxx(k) = int(n_block_start * w(i)) + 1
            k = k + 1
            if (k > n) exit out1
            do j = 1, l - 1
               indxx(k) = indxx(k - 1) + 1
               k = k + 1
               if (k > n) exit out1
            end do
         end do out1
      end if
      x_resample(:) = x(indxx(:))
      y_resample(:) = y(indxx(:))
!
! Test
!
!
!	  print *,' l = ',l
!	  print *,' i    x    y    x*    y*'
!     do i = 1, n
!        print '(4x, i6, 4(4x, f12.3))', i,x(i),y(i),x_resample(i),y_resample(i)
!     end do
!     read (*,*)
!
end subroutine bootstrap1
!
! ==================================================================================================
!    
function brent(ax, bx, cx, func, tol, xmin)
      use nrtype; use nrutil, only : nrerror
      implicit none
      real(sp), intent(in) :: ax, bx, cx, tol
      real(sp), intent(out) :: xmin
      real(sp) :: brent
      interface
          function func(x)
          use nrtype
          implicit none
          real(sp), intent(in) :: x
          real(sp) :: func
          end function func
      end interface
      integer(i4b), parameter :: itmax = 100
      real(sp), parameter :: cgold = 0.3819660_sp, zeps = 1.0e-3_sp*epsilon(ax)
      integer(i4b) :: iter
      real(sp) :: a, b, d, e, etemp, fu, fv, fw, fx, p, q, r, tol1, tol2, u, v, w, x, xm
      a = min(ax, cx)
      b = max(ax, cx)
      v = bx
      w = v
      x = v
      e = 0.0
      fx = func(x)
      fv = fx
      fw = fx
      do iter = 1, itmax
            xm = 0.5_sp*(a+b)
            tol1 = tol*abs(x)+zeps
            tol2 = 2.0_sp*tol1
            if (abs(x-xm) <= (tol2-0.5_sp*(b-a))) then
                  xmin = x
                  brent = fx
                  return
            end if
            if (abs(e) > tol1) then
                  r = (x-w)*(fx-fv)
                  q = (x-v)*(fx-fw)
                  p = (x-v)*q-(x-w)*r
                  q = 2.0_sp*(q-r)
                  if (q > 0.0) p = -p
                  q = abs(q)
                  etemp = e
                  e = d
                  if (abs(p) >= abs(0.5_sp*q*etemp) .or. &
                        p <= q*(a-x) .or. p >= q*(b-x)) then
                        e = merge(a-x, b-x,  x >= xm )
                        d = cgold*e
                  else
                        d = p/q
                        u = x+d
                        if (u-a < tol2 .or. b-u < tol2) d = sign(tol1, xm-x)
                  end if
            else
                  e = merge(a-x, b-x,  x >= xm )
                  d = cgold*e
            end if
            u = merge(x+d, x+sign(tol1, d),  abs(d) >= tol1 )
            fu = func(u)
            if (fu <= fx) then
                  if (u >= x) then
                        a = x
                  else
                        b = x
                  end if
                  call shft(v, w, x, u)
                  call shft(fv, fw, fx, fu)
            else
                  if (u < x) then
                        a = u
                  else
                        b = u
                  end if
                  if (fu <= fw .or. w == x) then
                        v = w
                        fv = fw
                        w = u
                        fw = fu
                  else if (fu <= fv .or. v == x .or. v == w) then
                        v = u
                        fv = fu
                  end if
            end if
      end do
      call nrerror('function brent: exceed maximum iterations')
contains
subroutine shft(a, b, c, d)
      real(sp), intent(out) :: a
      real(sp), intent(inout) :: b, c
      real(sp), intent(in) :: d
      a = b
      b = c
      c = d
end subroutine shft
end function brent
!
! ==================================================================================================
!
function brenttau(ax,bx,cx,lstau,tol,xmin,xfunc,yfunc,nfunc)
      use nrtype; use nrutil, only : nrerror
      implicit none
!     Brent's minimization, modified for persistence time estimation (subroutines minls, tauest).
      integer, intent(in) :: nfunc
      real(sp), dimension(nfunc) :: xfunc
      real(sp), dimension(nfunc) :: yfunc
      real(sp), intent(in) :: ax,bx,cx,tol
      real(sp), intent(out) :: xmin
      real(sp) :: brenttau
      interface
          function lstau(a,t,x,n)
          use nrtype
          implicit none
          integer, intent(in) :: n
          real(sp), dimension(n), intent(in) :: t
          real(sp), dimension(n), intent(in) :: x
          real(sp), intent(in) :: a
          real(sp) :: lstau
          end function lstau
      end interface
      integer(i4b), parameter :: itmax=100
      real(sp), parameter :: cgold=0.3819660_sp,zeps=1.0e-3_sp*epsilon(ax)
      integer(i4b) :: iter
      real(sp) :: a,b,d,e,etemp,fu,fv,fw,fx,p,q,r,tol1,tol2,u,v,w,x,xm
      a=min(ax,cx)
      b=max(ax,cx)
      v=bx
      w=v
      x=v
      e=0.0
      fx=lstau(x,xfunc,yfunc,nfunc)
      fv=fx
      fw=fx
      do iter=1,itmax
         xm=0.5_sp*(a+b)
         tol1=tol*abs(x)+zeps
         tol2=2.0_sp*tol1
         if (abs(x-xm) <= (tol2-0.5_sp*(b-a))) then
            xmin=x
            brenttau=fx
            return
         end if
         if (abs(e) > tol1) then
            r=(x-w)*(fx-fv)
            q=(x-v)*(fx-fw)
            p=(x-v)*q-(x-w)*r
            q=2.0_sp*(q-r)
            if (q > 0.0) p=-p
            q=abs(q)
            etemp=e
            e=d
            if (abs(p) >= abs(0.5_sp*q*etemp) .or.                      &
                p <= q*(a-x) .or. p >= q*(b-x)) then
               e=merge(a-x,b-x, x >= xm )
               d=cgold*e
            else
               d=p/q
               u=x+d
               if (u-a < tol2 .or. b-u < tol2) d=sign(tol1,xm-x)
            end if
         else
            e=merge(a-x,b-x, x >= xm )
            d=cgold*e
         end if
         u=merge(x+d,x+sign(tol1,d), abs(d) >= tol1 )
         fu=lstau(u,xfunc,yfunc,nfunc)
         if (fu <= fx) then
            if (u >= x) then
               a=x
            else
               b=x
            end if
            call shft(v,w,x,u)
            call shft(fv,fw,fx,fu)
         else
            if (u < x) then
               a=u
            else
               b=u
            end if
            if (fu <= fw .or. w == x) then
               v=w
               fv=fw
               w=u
               fw=fu
            else if (fu <= fv .or. v == x .or. v == w) then
               v=u
               fv=fu
            end if
         end if
      end do
      call nrerror('brenttau: exceed maximum iterations')
contains
!
subroutine shft(a,b,c,d)
      real(sp), intent(out) :: a
      real(sp), intent(inout) :: b,c
      real(sp), intent(in) :: d
      a=b
      b=c
      c=d
end subroutine shft
end function brenttau
!
! ==================================================================================================
!
subroutine calc_inv_alpha
      use nrtype
      use parameters, only: alpha
      use setting, only: phi_inv_alpha
      implicit none
!     Calculates phi_inv_alpha, the percentage point of the standard Gaussian distribution.
      real(sp) :: phi_inv
      phi_inv_alpha = phi_inv(alpha)
!
! test
!     print *,'phi_inv_alpha: ', phi_inv_alpha
!
end subroutine calc_inv_alpha
!
! ==================================================================================================
!
subroutine calc_inv_student
      use nrtype
  	  use setting, only: n, t_ave_alpha
      implicit none
!     Calculates t_ave_alpha, the percentage point of Student's t distribution with
!     n - 2 degrees of freedom (since linear regression involves 2 parameters, beta0 and beta1).
      real(sp) :: t_inv
      t_ave_alpha = t_inv(n - 2)
!
! test
!     print *, ' t_ave_alpha',  t_ave_alpha
!
end subroutine calc_inv_student
!
! ==================================================================================================
!
subroutine chprior1
      use nrtype
      use prior_knowledge, only: sx_const, sy_const, delta_const, sx, sy, delta
      use parameters, only: ntry
      use setting, only: datacase, n
      implicit none
!     Changes prior knowledge: data errors.
      integer :: i
      real(sp), parameter ::  one = 1.0_sp
      real(sp), parameter :: zero = 0.0_sp
l1:   do i = 1, ntry
         print *
         if (datacase .eq. 1 .or. datacase .eq. 2) then
            print '(a)',                                                                        &
' standard deviation (constant) of data x(i)                                      [INPUT]'
         else if (datacase .eq. 3 .or. datacase .eq. 4) then
            exit l1
         end if   
         read (5,*) sx_const
         sx_const = sx_const * one
         if (sx_const .gt. zero ) then
            exit l1
         else
            if (i < ntry) then
               print '(a)',' Error since sx_const has to be positive - try again.'
            else
               print '(a)',' Too many invalid attempts - LINCAL terminates.'
               stop
            end if
         end if
      end do l1
!
l2:   do i = 1, ntry
         print *
         if (datacase .eq. 1 .or. datacase .eq. 2) then
            print '(a)',                                                                        &
' standard deviation (constant) of data y(i)                                      [INPUT]'
         else if (datacase .eq. 3 .or. datacase .eq. 4) then
            exit l2
         end if   
         read (5,*) sy_const
         sy_const = sy_const * one
         if (sy_const .gt. zero ) then
            exit l2
         else
            if (i < ntry) then
               print '(a)',' Error since sy_const has to be positive - try again.'
            else
               print '(a)',' Too many invalid attempts - LINCAL terminates.'
               stop
            end if
         end if
      end do l2
      if (datacase .eq. 1 .or. datacase .eq. 2) then
         delta_const = sx_const / sy_const
      else if (datacase .eq. 3 .or. datacase .eq. 4) then
         delta(1:n) = sx(1:n) / sy(1:n)
      end if      
end subroutine chprior1
!
! ==================================================================================================
!
subroutine chprior2
      use nrtype
      use prior_knowledge, only: sx_predict
      use parameters, only: ntry
      use setting, only: datacase
      implicit none
!     Changes prior knowledge: prediction data errors.
      integer :: i
      real(sp), parameter ::  one = 1.0_sp
      real(sp), parameter :: zero = 0.0_sp
l1:   do i = 1, ntry
         print *
         if (datacase .eq. 1 .or. datacase .eq. 2) then
            exit l1
         else if (datacase .eq. 3 .or. datacase .eq. 4) then
            print '(a)',                                                                        &
' standard deviation (constant) of data x(i) for prediction                       [INPUT]'
         end if
         read (5,*) sx_predict
         sx_predict = sx_predict * one
         if (sx_predict .gt. zero ) then
            exit l1
         else
            if (i < ntry) then
               print '(a)',' Error since standard deviation has to be positive - try again.'
            else
               print '(a)',' Too many invalid attempts - LINCAL terminates.'
               stop
            end if
         end if
      end do l1
end subroutine chprior2
!
! ==================================================================================================
!
subroutine chsett1
      use parameters, only: ntry
      use setting, only: datacase
      implicit none
!     Changes setting: datacase.
      integer :: i
      print *
      print '(a)',' Select the situation of input data and prior knowledge'
      print '(a)',' ======================================================'
      print *
      print '(a)',' Minimum requirement for the calibration is the existence of an'
      print '(a)',' input predictor (x) and input response (y) data sample.'
      print '(a)',' We write x(i) and y(i), where the counter, i,'
      print '(a)',' runs from 1 to n (sample size).'
      print *
      print '(a)',' It may be possible that x(i) and y(i) stem from time series data;'
      print '(a)',' then an additional requirement is the existence of'
      print '(a)',' time data, t(i).'
      print *
      print '(a)',' In addition to the sample, we need prior knowledge about data errors.'
      print '(a)',' The simplest situation (homoscedasticity) for the error type is if'
      print '(a)',' the x(i) have constant standard deviation, sx.'
      print '(a)',' Then no knowledge about data errors of y(i) is required.'
      print *
      print '(a)',' A more advanced situation (heteroscedasticity) is if'
      print '(a)',' both x(i) and y(i) have nonconstant standard deviations, sx(i) and sy(i).'
      print *
      print '(a)',' Each of the four cases requires a designed combination of'
      print '(a)',' parameter estimation and uncertainty determination.'
      print '(a)',' Please see the book (Mudelsee 2014, Chapter 8 therein).'
      print *
l1:   do i = 1, ntry
         print *
         print '(a)',                                                                           &
         ' Case    Data                                    Prior knowledge        '
         print *
         print '(a)',                                                                           &
         ' Case 1: {      x(i), y(i)}, i = 1, ..., n,      sx (constant), sy (constant)    [1]'
         print '(a)',                                                                           &
         ' Case 2: {t(i), x(i), y(i)}, i = 1, ..., n,      sx (constant), sy (constant)    [2]'
         print '(a)',                                                                           &
         ' Case 3: {      x(i), y(i),                      sx(i), sy(i)}, i = 1, ..., n    [3]'
         print '(a)',                                                                           &
         ' Case 4: {t(i), x(i), y(i),                      sx(i), sy(i)}, i = 1, ..., n    [4]'
         read (5,*) datacase
         if (datacase .eq. 1 .or. datacase .eq. 2 .or.                                          &
             datacase .eq. 3 .or. datacase .eq. 4) then
            exit l1
         else
            if (i < ntry) then
               print '(a)',' Invalid input data case - try again.'
            else
               print '(a)',' Too many invalid attempts - LINCAL terminates.'
               stop
            end if
         end if
      end do l1
end subroutine chsett1
!
! ==================================================================================================
!
subroutine chsett2
      use parameters, only: ntry
      use setting, only: datacase, datafile
      implicit none
!      
!     Changes setting: datafile.
!     Note: data files may have header lines (defined by first character = '#').
!
      integer :: i, open_error
l1:   do i = 1, ntry
         print *
         if (datacase .eq. 1) then
            print '(a)',                                                                        &
' data (x, y)                                                                     [path + filename]'
         else if (datacase .eq. 2) then
            print '(a)',                                                                        &
' data (t, x, y)                                                                  [path + filename]'
         else if (datacase .eq. 3) then
            print '(a)',                                                                        &
' data (x, y, sx, sy)                                                             [path + filename]'
         else if (datacase .eq. 4) then
            print '(a)',                                                                        &
' data (t, x, y, sx, sy)                                                          [path + filename]'
         end if   
         read (5,'(a)') datafile
         open (unit = 1, file = datafile, status = 'old',                                       &
               form = 'formatted', action = 'read', iostat = open_error)
         if (open_error .eq. 0 ) then
            exit l1
         else
            if (i < ntry) then
               print '(a)',' Error during data file opening - try again.'
            else
               print '(a)',' Too many invalid attempts - LINCAL terminates.'
               stop
            end if
         end if
      end do l1
      close (unit = 1, status = 'keep')
end subroutine chsett2
!
! ==================================================================================================
!
subroutine chsett3
      use nrtype
      use parameters, only: nmin
      use setting, only: datacase, datafile, n
      implicit none
!     Changes setting: n.
      character (len = 1) :: flag
      integer :: i, iocheck, open_error
      real(sp) :: tdum  = -999.0_sp ! dummy
      real(sp) :: xdum  = -999.0_sp ! dummy
      real(sp) :: ydum  = -999.0_sp ! dummy
      real(sp) :: sxdum = -999.0_sp ! dummy
      real(sp) :: sydum = -999.0_sp ! dummy
      open (unit = 1, file = datafile, status = 'old',                                          &
            form = 'formatted', action = 'read', iostat = open_error)
      if (open_error /= 0 ) then
         print '(a)',' Error during data file opening - LINCAL terminates.'
         stop
      end if
!
! 1.  Skip header
!     ===========
!
      do while (.true.)
         read (1, '(a1)') flag
         if (flag .ne. '#') then
            backspace (1)
            exit
         end if
      end do
!
! 2.  Count data
!     ==========
!
      i = 1
      do while (.true.)
         if (datacase .eq. 1) then
             read (1, *, iostat = iocheck) xdum, ydum
         else if (datacase .eq. 2) then
             read (1, *, iostat = iocheck) tdum, xdum, ydum
         else if (datacase .eq. 3) then
             read (1, *, iostat = iocheck) xdum, ydum, sxdum, sydum
         else if (datacase .eq. 4) then
             read (1, *, iostat = iocheck) tdum, xdum, ydum, sxdum, sydum
         end if
         if (iocheck .ne. 0) exit
         i = i + 1
      end do
      close (unit = 1, status = 'keep')
      n = i - 1
      if (n < nmin) then
         print '(a,i2)',' sample size (original) : ', n
         print '(a,i2)',' minimum required is    : ', nmin
         print *
         print '(a)',' LINCAL terminates.'
         stop
      end if
end subroutine chsett3
!
! ==================================================================================================
!
subroutine clearscreen
      implicit none
!     Clears screen.
      integer :: i
      integer, parameter :: n = 120
      do i = 1, n
         print *
      end do
end subroutine clearscreen
!
! ==================================================================================================
!
subroutine confidence1(rng3, rng4)
      use nrtype
      use rng
      use data1, only: t, x, y, res_x, res_y
      use parameters, only: b, np
      use prior_knowledge, only: delta_const, delta, sx, sy, sx_const, sx_predict
      use result1, only: beta0    , beta1    , xp, yp,                                          &
                         beta0_low, beta1_low,     yp_low,                                      &
                         beta0_upp, beta1_upp,     yp_upp
      use setting, only: datacase, l_mbb, n, tau_x, tau_y, t_ave_alpha
      implicit none
      type(rng_t), dimension(1), intent(inout) :: rng3          ! seeded random number generator
      type(rng_t), dimension(1), intent(inout) :: rng4          ! seeded random number generator
!
!     Determines bootstrap Student's t confidence intervals using pairwise-MBBres resampling.
!
      integer :: i = -999                                       ! counter
      integer :: ierror = -999                                  ! error flag (allocate, deallocate)
      integer :: iseed = 1                                      ! dummy integer for random number generator
      integer :: k = -999                                       ! counter, bootstrap resampling
      real(sp), allocatable, dimension(:, :) :: x_resample      ! resample x-data
      real(sp), allocatable, dimension(:, :) :: y_resample      ! resample y-data
      real(sp), dimension(1:n) :: x_fit                         ! fit value, x
      real(sp), dimension(1:n) :: y_fit                         ! fit value, y
      real(sp), dimension(1:b) :: beta0_resample                ! replication of beta0
      real(sp), dimension(1:b) :: beta1_resample                ! replication of beta1
      real(sp), allocatable, dimension(:, :) ::    yp_resample  ! replication of yp
      real(sp), allocatable, dimension(:) ::    se_yp_resample  ! bootstrap standard error of yp_resample
      real(sp), allocatable, dimension(:) ::        x_gauss     ! standard Gaussian variate
      real(sp) :: ax                  = -999.0_sp               ! equivalent autocorrelation coefficient, res_x
      real(sp) :: ay                  = -999.0_sp               ! equivalent autocorrelation coefficient, res_y
      real(sp) :: axy                 = -999.0_sp               ! combined equivalent autocorrelation coefficient
      real(sp) :: avex                = -999.0_sp               ! average,  res_x
      real(sp) :: avey                = -999.0_sp               ! average,  res_y
      real(sp) :: varx                = -999.0_sp               ! variance, res_x
      real(sp) :: vary                = -999.0_sp               ! variance, res_y
      real(sp) :: se_beta0_resample   = -999.0_sp               ! bootstrap standard error of beta0_resample
      real(sp) :: se_beta1_resample   = -999.0_sp               ! bootstrap standard error of beta1_resample
      real(sp) :: dummy1              = -999.0_sp
      real(sp) :: dummy2              = -999.0_sp
      real(sp) :: dummy3              = -999.0_sp
      real(sp), parameter :: zero     = 0.0_sp
      real(sp), parameter :: onethird = 0.3333333333333333333333333333333333333333_sp
      real(sp), parameter :: half     = 0.5_sp
      real(sp), parameter :: twothird = 0.6666666666666666666666666666666666666666_sp
      real(sp), parameter :: one      = 1.0_sp
      real(sp), parameter :: two      = 2.0_sp
      real(sp), parameter :: six      = 6.0_sp
!
! 1.  Allocate and initialize resample data and replications
!     ======================================================
!
      allocate(x_resample(1:b, 1:n), y_resample(1:b, 1:n), yp_resample(1:b, 1:np),              &
               se_yp_resample(1:np), x_gauss(1:np), stat = ierror)
      if (ierror /= 0) then
         print '(a)',' Subroutine confidence1: unexpected allocation error - LINCAL terminates.'
         stop
      end if      
      x_resample     = -999.0_sp
      y_resample     = -999.0_sp
      yp_resample    = -999.0_sp
      se_yp_resample = -999.0_sp
      x_gauss        = -999.0_sp
      beta0_resample = -999.0_sp
      beta1_resample = -999.0_sp
!
! 2.  Residuals and fit values
!     ========================
!
!     Note: we treat here also heteroscedastic data errors, expressed via delta(:), while
!           the Monte Carlo simulations in the book (Mudelsee 2014) only consider
!           homoscedastic errors.
!
      if (datacase .eq. 1 .or. datacase .eq. 2) then
         res_x(:) = (beta0 + beta1 * x(:) - y(:)) / (delta_const**two / beta1 + beta1)
         res_y(:) = -res_x(:) * delta_const**two / beta1
      else if (datacase .eq. 3 .or. datacase .eq. 4) then
         res_x(:) = (beta0 + beta1 * x(:) - y(:)) / (delta(:)**two / beta1 + beta1)
         res_y(:) = -res_x(:) * delta(:)**two / beta1
      end if
      x_fit(:) = x(:) - res_x(:)
      y_fit(:) = y(:) - res_y(:)
!
! 3.  Persistence times and block length selection
!     ============================================
!
!     Notes: (1) For absent time values (datacase = 1 or 3), no persistence is assumed.
!            (2) In the pairwise-MBBres algorithm, tau_x = tau_y, but we test this here
!                by estimating and outputting both.
!            (3) The block length is selected via the adapted Carlstein formula for the
!                bivariate setting, see the book (Mudelsee 2014, Eqs. 7.31 and 7.32 therein).
!
      if (datacase .eq. 1 .or. datacase .eq. 3) then
         l_mbb = 1
      else if (datacase .eq. 2 .or. datacase .eq. 4) then
         call avevar_n(res_x, n, avex, varx)
         call avevar_n(res_y, n, avey, vary)
         call tauest(t, (res_x(:) - avex) / sqrt(varx), n, tau_x, ax)
         call tauest(t, (res_y(:) - avey) / sqrt(vary), n, tau_y, ay)
!
! Test 2024 --- Start
!
!        print *,' TEST 2024 --- tau_x = ',tau_x
!        print *,' TEST 2024 --- tau_y = ',tau_y
!
!        open  (unit = 1, file = 'res_x.dat', status = 'unknown', form = 'formatted', action = 'write')
!        do i = 1, n
!           write (unit = 1, fmt = '(3x, f16.8, 3x, f16.8)'), t(i), res_x(i)
!        end do
!        close (unit = 1, status = 'keep')
!        print *, ' res_x dataset has been written'
!        print *
!
!        open  (unit = 1, file = 'res_y.dat', status = 'unknown', form = 'formatted', action = 'write')
!        do i = 1, n
!           write (unit = 1, fmt = '(3x, f16.8, 3x, f16.8)'), t(i), res_y(i)
!        end do
!        close (unit = 1, status = 'keep')
!        print *, ' res_y dataset has been written'
!        print *
!
!        read (*,*)
!
! Test 2024 --- End
!
         axy    = sqrt(ax * ay)
         dummy1 = six**half * axy
         dummy2 = one - axy**two
         dummy3 = (dummy1 / dummy2)**twothird * n**onethird
         l_mbb  = anint(dummy3)
         l_mbb  = max(1,     l_mbb)
         l_mbb  = min(n - 1, l_mbb)
      end if
!
! 4.  Bootstrap CIs
!     =============
!
!
! 4.1 Start of loop
!     =============
!
      do k = 1, b
!
! 4.2    Resampling
!        ==========
!
         call bootstrap1(n, res_x(1:n), res_y(1:n), l_mbb,                                      &
                         x_resample(k, 1:n), y_resample(k, 1:n), rng3(iseed),rng4(iseed))
         x_resample(k, 1:n) = x_resample(k, 1:n) + x_fit(1:n)
         y_resample(k, 1:n) = y_resample(k, 1:n) + y_fit(1:n)
!
! 4.3    Replications of parameter estimates
!        ===================================
!
          if (datacase .eq. 1 .or. datacase .eq. 2) then
             call olsbc(x_resample(k, 1:n), y_resample(k, 1:n), n, sx_const, beta0_resample(k), beta1_resample(k))
          else if (datacase .eq. 3 .or. datacase .eq. 4) then
             call wlsxy(x_resample(k, 1:n), y_resample(k, 1:n), sx, sy, n,   beta0_resample(k), beta1_resample(k))
          end if
!
! 4.4    Replications of prediction
!        ==========================
!
          do i = 1, np
             call gaussian(x_gauss(i), rng3(iseed), rng4(iseed))
          end do
!
! Test
!
!         open  (unit = 1, file = 'gaussiandata.dat', status = 'unknown', form = 'formatted', action = 'write')
!         do i = 1, 100000
!            call gaussian(dummy1, rng3(iseed), rng4(iseed))
!            write (unit = 1, fmt = '(3x, i7, 3x, f16.8)'), i, dummy1
!         end do
!         close (unit = 1, status = 'keep')
!         print *, ' Gaussian test dataset has been written'
!         read (*,*)
!
          if (datacase .eq. 1 .or. datacase .eq. 2) then
             call linmod(np, xp(:) + sx_const   * x_gauss(:), beta0_resample(k), beta1_resample(k), yp_resample(k, 1:np))
          else if (datacase .eq. 3 .or. datacase .eq. 4) then
             call linmod(np, xp(:) + sx_predict * x_gauss(:), beta0_resample(k), beta1_resample(k), yp_resample(k, 1:np))
          end if
!
! Test (below call would correspond to a prediction with ignored predictor noise)
!
!         call linmod(np, xp, beta0_resample(k), beta1_resample(k), yp_resample(k, 1:np))
!
! 4.5 End of loop
!     ===========
!
      end do
!
! 4.6 Bootstrap standard errors
!     =========================
!
      call avevar_n(beta0_resample(1:b), b, dummy1, dummy2)
      se_beta0_resample = sqrt(dummy2)
      call avevar_n(beta1_resample(1:b), b, dummy1, dummy2)
      se_beta1_resample = sqrt(dummy2)
      do i = 1, np
         call avevar_n(yp_resample(1:b, i), b, dummy1, dummy2)
         se_yp_resample(i) = sqrt(dummy2)
      end do
!
! 4.7 Student's t CIs
!     ===============
!
      beta0_low = beta0 + t_ave_alpha * se_beta0_resample
      beta0_upp = beta0 - t_ave_alpha * se_beta0_resample
      beta1_low = beta1 + t_ave_alpha * se_beta1_resample
      beta1_upp = beta1 - t_ave_alpha * se_beta1_resample
      do i = 1, np
         yp_low(i) = yp(i) + t_ave_alpha * se_yp_resample(i)
         yp_upp(i) = yp(i) - t_ave_alpha * se_yp_resample(i)
      end do
!
! Test
!
!
!     open  (unit = 9, file = 'replicationtest.dat', status = 'unknown', form = 'formatted', action = 'write')
!     do k = 1, b
!        write (unit = 9, fmt = '(3x, i7, 2(3x, f16.8))'), k, beta0_resample(k), beta1_resample(k)
!     end do
!     close (unit = 9, status = 'keep')
!     print *, ' Replication test dataset has been written'
!     read (*,*)
!
! 5.  Deallocate resample data
!     ========================
!
      deallocate(x_resample, y_resample, yp_resample, se_yp_resample, x_gauss, stat = ierror)
      if (ierror /= 0) then
         print '(a)',' Subroutine confidence1: unexpected deallocation error - LINCAL terminates.'
         stop
      end if
end subroutine confidence1
!
! ==================================================================================================
!
subroutine deallocate1
      use nrtype
      use prior_knowledge, only: sx, sy
      use setting, only: datacase
      use data1, only: t, x, y
      implicit none
!     Deallocates data.
      integer :: error = 0
      if (datacase .eq. 1) then
         deallocate(x, y, stat = error)
      else if (datacase .eq. 2) then   
         deallocate(t, x, y, stat = error)
      else if (datacase .eq. 3) then
         deallocate(x, y, sx, sy, stat = error)
      else if (datacase .eq. 4) then   
         deallocate(t, x, y, sx, sy, stat = error)
      end if
      if (error /= 0) then
         print '(a)',' Subroutine deallocate1: unexpected deallocation error - LINCAL terminates.'
         stop
      end if
end subroutine deallocate1
!
! ==================================================================================================
!
subroutine estimation1
      use nrtype
      use data1, only: t, x, y
      use prior_knowledge, only: sx, sx_const, sy
      use result1, only: beta0, beta1
      use setting, only: datacase, n
      implicit none
!
!     Estimates beta0, beta1.
!     Note: sy_const (datacase = 1 or 2) is not required for OLSBC estimation but for
!           OLSBC uncertainty determination (Mudelsee 2014).
!
      real(sp), parameter ::  one = 1.0_sp
      real(sp), parameter :: zero = 0.0_sp
      if (datacase .eq. 1 .or. datacase .eq. 2) then
         call olsbc(x, y, n, sx_const, beta0, beta1)
      else if (datacase .eq. 3 .or. datacase .eq. 4) then
         call wlsxy(x, y, sx, sy, n, beta0, beta1)
      end if   
end subroutine estimation1
!
! ==================================================================================================
!
subroutine gaussian(z, rng3, rng4)
      use nrtype
      use rng
      implicit none
      real(sp), intent(out) :: z
      type(rng_t), dimension(1), intent(inout) :: rng3      ! seeded random number generator
      type(rng_t), dimension(1), intent(inout) :: rng4      ! seeded random number generator
!
!     Gaussian standard distribution using the transformation after Box and Muller (1958).
!
!     Reference:
!
!     Box GEP, Muller ME (1958) A note on the generation of random normal deviates.
!                               Annals of Mathematical Statistics 29:610-611.
!
      real(sp) :: z1
      real(sp) :: z2
      real(sp) :: zsq
      real(sp), parameter :: zero = 0.0_sp
      real(sp), parameter ::  one = 1.0_sp
      real(sp), parameter ::  two = 2.0_sp
      integer :: iseed = 1                                  ! dummy integer for random number generator
      do
         z1  = mzran(rng3(iseed))
         z2  = mzran(rng4(iseed))
         z1  = two * z1 - one
         z2  = two * z2 - one
         zsq = z1**two + z2**two
         if (zsq > zero .and. zsq < one) exit
      end do
      zsq = sqrt(-two * log(zsq) / zsq)
      z = z1 * zsq
end subroutine gaussian
!
! ==================================================================================================
!
subroutine info1
      implicit none
!     Welcomes.
      integer :: i
      integer, parameter :: n = 120
      do i = 1, n
         print '(a)',' '
      end do
      print '(a)',     ' ==========================================================================='
      print '(a)',     '                                                                            '
      print '(a,a,a)', '  LINCAL (Version 1.00',',',' December 2022)                                '
      print '(a)',     '                                                                            '
      print '(a)',     '  A software for calibrating linear climate proxy models                    '
      print '(a)',     '                                                                            '
      print '(a)',     ' ==========================================================================='
      print '(a)',     '                                                                            '
      print '(a)',     '  References:                                                               '
      print '(a)',     '                                                                            '
      print '(a)',     '  Mudelsee M (2023) Unbiased proxy calibration.                             '
      print '(a)',     '                    Mathematical Geosciences.                               '
      print '(a)',     '                    doi:10.1007/s11004-023-10122-5.                         '
      print '(a)',     '                    [published online 19 December 2023]                     '
      print '(a)',     '                                                                            '
      print '(a)',     '  Mudelsee M (2014) Climate Time Series Analysis:                           '
      print '(a)',     '                    Classical Statistical and Bootstrap Methods.            '
      print '(a)',     '                    Second Edition. Springer, Cham, Switzerland, 454 pp.    '
      print '(a)',     '                                                                            '
      print '(a)',     '  Copyright (C) Manfred Mudelsee                                            '
      print '(a)',     '  https://www.manfredmudelsee.com                                           '
      print '(a)',     '                                                                            '
      print '(a)',     '  Please see source code (LINCAL.f90) for                                   '
      print '(a)',     '  details, disclaimer and permissions.                                      '
      print '(a)',     '                                                                            '
      print '(a)',     '  PRESS ENTER TO CONTINUE                                                   '
      print '(a)',     '                                                                            '
      read (*,*)
end subroutine info1
!
! ==================================================================================================
!    
subroutine info2
      use parameters, only: outputfile
      implicit none
!     Says goodbye.
      integer :: i
      integer, parameter :: n = 3
      do i = 1, 3
         print '(a)',' '
      end do
      print '(a,a,a)', ' LINCAL terminated successfully.                                            '
      print '(a)',     '                                                                            '
      print '(a,a)',   ' LINCAL outputfile: ', outputfile
      print '(a)',     '                                                                            '
      print '(a)',     ' ==========================================================================='
      print '(a)',     '                                                                            '
end subroutine info2
!
! ==================================================================================================
!
subroutine init1
      use nrtype
      use data1, only: t, x, y, res_x, res_y
      use prior_knowledge, only: sx, sy, delta
      use setting, only: datacase
      implicit none
!     Initializes data, residuals.
      if (datacase .eq. 1) then
         x     = -999.0_sp
         y     = -999.0_sp
      else if (datacase .eq. 2) then
         t     = -999.0_sp
         x     = -999.0_sp
         y     = -999.0_sp
      else if (datacase .eq. 3) then
         x     = -999.0_sp
         y     = -999.0_sp
         sx    = -999.0_sp
         sy    = -999.0_sp
         delta = -999.0_sp
      else if (datacase .eq. 4) then
         t     = -999.0_sp
         x     = -999.0_sp
         y     = -999.0_sp
         sx    = -999.0_sp
         sy    = -999.0_sp
         delta = -999.0_sp
      end if
      res_x = -999.0_sp
      res_y = -999.0_sp
end subroutine init1
!
! ==================================================================================================
!
subroutine linmod(np, xp, beta0, beta1, yp)
      use nrtype
      implicit none
      integer, intent(in) :: np                     ! sample size (prediction)
      real(sp), dimension(np), intent(in) ::  xp    ! x-data (prediction)
      real(sp), dimension(np), intent(out) :: yp    ! y-data (prediction)
      real(sp), intent(in) :: beta0                 ! intercept
      real(sp), intent(in) :: beta1                 ! slope
!
!     Linear prediction model for yp on basis of input data xp.
!
      yp(1:np) = beta0 + beta1 * xp(1:np)
end subroutine linmod
!
! ==================================================================================================
!
function lstau(a,t,x,n)
      use nrtype
      implicit none
      integer, intent(in) :: n
      real(sp), dimension(n), intent(in) :: t
      real(sp), dimension(n), intent(in) :: x
      real(sp), intent(in) :: a
!     Least-squares function for tau estimation.
      real(sp) :: lstau
      integer :: i
      lstau=0.0_sp
      do i=2,n
         lstau=lstau+(x(i)-x(i-1)*sign(1.0_sp,a)*abs(a)**(t(i)-t(i-1)))**2.0_sp
      end do
end function lstau
!
! ==================================================================================================
!
subroutine minls(n, t, x, amin, nmu_)
      use nrtype
      implicit none
      integer, intent(in) :: n
      real(sp), dimension(n), intent(in) :: t
      real(sp), dimension(n), intent(in) :: x
      real(sp), intent(out) :: amin
      integer, intent(out)  :: nmu_
!     Minimizes least-squares function s using Brent's method.
      real(sp), parameter :: a_ar1 = 0.3678794_sp  ! 1/e
      real(sp), parameter :: tol   = 3.0e-08_sp    ! Brent's search, precision
      real(sp), parameter :: tol2  = 1.0e-06_sp    ! multiple solutions, precision
      real(sp) :: dum1   = -999.0_sp
      real(sp) :: dum2   = -999.0_sp
      real(sp) :: dum3   = -999.0_sp
      real(sp) :: dum4   = -999.0_sp
      real(sp) :: a_ar11 = -999.0_sp
      real(sp) :: a_ar12 = -999.0_sp
      real(sp) :: a_ar13 = -999.0_sp
      real(sp) :: brenttau
      interface
          function lstau(a, t, x, n)
          use nrtype
          implicit none
          integer, intent(in) :: n
          real(sp), dimension(n), intent(in) :: t
          real(sp), dimension(n), intent(in) :: x
          real(sp), intent(in) :: a
          real(sp) :: lstau
          end function lstau
      end interface
      nmu_ = 0
      dum1 = brenttau(-2.0_sp,           a_ar1          , +2.0_sp,            &
                      lstau, tol, a_ar11, t, x, n)
      dum2 = brenttau(  a_ar1, 0.5_sp * (a_ar1 + 1.0_sp), +2.0_sp,            &
                      lstau, tol, a_ar12, t, x, n)
      dum3 = brenttau(-2.0_sp, 0.5_sp * (a_ar1 - 1.0_sp),   a_ar1,            &
                      lstau, tol, a_ar13, t, x, n)
      if ((abs(a_ar12 - a_ar11) > tol2   .and.                                &
           abs(a_ar12 - a_ar1)  > tol2)  .or.                                 &
          (abs(a_ar13 - a_ar11) > tol2   .and.                                &
           abs(a_ar13 - a_ar1)  > tol2))                                      &
         nmu_ = 1
      dum4 = min(dum1, dum2, dum3)
      if (dum4 == dum2) then
         amin = a_ar12
      else if (dum4 == dum3) then
         amin = a_ar13
      else
         amin = a_ar11
      end if
end subroutine minls
!
! ==================================================================================================
!    
subroutine mmbrak(ax, bx, cx, fa, fb, fc, func)
      use nrtype
      use nrutil, only : swap
      implicit none
      real(sp), intent(inout) :: ax, bx
      real(sp), intent(out) :: cx, fa, fb, fc
      interface
          function func(x)
          use nrtype
          implicit none
          real(sp), intent(in) :: x
          real(sp) :: func
          end function func
      end interface
      real(sp), parameter :: gold = 1.618034_sp, glimit = 100.0_sp, tiny = 1.0e-20_sp
      real(sp) :: fu, q, r, u, ulim
      fa = func(ax)
      fb = func(bx)
      if (fb > fa) then
            call swap(ax, bx)
            call swap(fa, fb)
      end if
      cx = bx+gold*(bx-ax)
      fc = func(cx)
      do
            if (fb < fc) return
            r = (bx-ax)*(fb-fc)
            q = (bx-cx)*(fb-fa)
            u = bx-((bx-cx)*q-(bx-ax)*r)/(2.0_sp*sign(max(abs(q-r), tiny), q-r))
            ulim = bx+glimit*(cx-bx)
            if ((bx-u)*(u-cx) > 0.0) then
                  fu = func(u)
                  if (fu < fc) then
                        ax = bx
                        fa = fb
                        bx = u
                        fb = fu
                        return
                  else if (fu > fb) then
                        cx = u
                        fc = fu
                        return
                  end if
                  u = cx+gold*(cx-bx)
                  fu = func(u)
            else if ((cx-u)*(u-ulim) > 0.0) then
                  fu = func(u)
                  if (fu < fc) then
                        bx = cx
                        cx = u
                        u = cx+gold*(cx-bx)
                        call shft(fb, fc, fu, func(u))
                  end if
            else if ((u-ulim)*(ulim-cx) >=  0.0) then
                  u = ulim
                  fu = func(u)
            else
                  u = cx+gold*(cx-bx)
                  fu = func(u)
            end if
            call shft(ax, bx, cx, u)
            call shft(fa, fb, fc, fu)
      end do
contains
subroutine shft(a, b, c, d)
      real(sp), intent(out) :: a
      real(sp), intent(inout) :: b, c
      real(sp), intent(in) :: d
      a = b
      b = c
      c = d
end subroutine shft
end subroutine mmbrak
!
! ==================================================================================================
!    
subroutine ols(x, y, n, beta0, beta1)
      use nrtype
      implicit none
      integer, intent(in) :: n                  ! sample size
      real(sp), dimension(n), intent(in) :: x   ! data
      real(sp), dimension(n), intent(in) :: y   ! data
      real(sp), intent(out) :: beta0            ! intercept
      real(sp), intent(out) :: beta1            ! slope
!
!     Ordinary least-squares (OLS) estimation of a linear regression model.
!
      real(sp), dimension(n) :: xwork
      real(sp) :: ss
      real(sp) :: stt
      real(sp) :: sx
      real(sp) :: sxss
      real(sp) :: sy
      real(sp), parameter :: one = 1.0_sp
      ss = n * one
      sx = sum(x)
      sy = sum(y)
      sxss = sx / ss
      xwork(1:n) = x(1:n) - sxss
      beta1 = dot_product(xwork, y)
      stt = dot_product(xwork, xwork)
      beta1 = beta1 / stt
      beta0 = (sy - sx * beta1) / ss
end subroutine ols
!
! ==================================================================================================
!    
subroutine olsbc(x, y, n, sx_const, beta0, beta1)
      use nrtype
      implicit none
      integer, intent(in) :: n                  ! sample size
      real(sp), dimension(n), intent(in) :: x   ! data
      real(sp), dimension(n), intent(in) :: y   ! data
      real(sp), intent(in) :: sx_const          ! standard deviation (constant) of data x
      real(sp), intent(out) :: beta0            ! intercept
      real(sp), intent(out) :: beta1            ! slope
!
!     Ordinary least-squares with bias correction (OLSBC) estimation of a linear regression model.
!
      real(sp) :: avex = -999.0_sp              ! average
      real(sp) :: varx = -999.0_sp              ! variance
      real(sp), parameter :: one = 1.0_sp
      real(sp), parameter :: two = 2.0_sp
!
! 1.  OLS
!     ===
!
      beta0 = -999.0_sp
      beta1 = -999.0_sp
      call ols(x, y, n, beta0, beta1)
!
! 2.  Bias correction
!     ===============
!
      call avevar_n(x, n, avex, varx)
      beta1 = beta1 / (one - sx_const**two / varx)
      beta0 = (sum(y) - beta1 * sum(x)) / n
end subroutine olsbc
!
! ==================================================================================================
!
subroutine output1
      use nrtype
      use parameters, only: alpha, b, np, outputfile
      use prior_knowledge, only: sx_const, sy_const, sx_predict
      use result1, only: beta0, beta0_low, beta0_upp, beta1, beta1_low, beta1_upp,              &
                         xp, yp, yp_low, yp_upp
      use setting, only: datacase, datafile, l_mbb, n, tau_x, tau_y
      implicit none
!
!     Outputs result.
!
      integer :: i
      real(sp), parameter :: one = 1.0_sp
      real(sp), parameter :: two = 2.0_sp

! 1.  Preliminary
!     ===========
!
      open  (unit = 1, file = outputfile, status = 'unknown', form = 'formatted', action = 'write')
      write (unit = 1, fmt = '(a)')     '                                                                            '
      write (unit = 1, fmt = '(a)')     ' ==========================================================================='
      write (unit = 1, fmt = '(a)')     '                                                                            '
      write (unit = 1, fmt = '(a)')     '  Results                                                                   '
      write (unit = 1, fmt = '(a)')     '                                                                            '
      write (unit = 1, fmt = '(a)')     ' ==========================================================================='
      write (unit = 1, fmt = '(a)')     '                                                                            '
      write (unit = 1, fmt = '(a,a,a)') '  LINCAL (Version 1.00',',',' December 2022)                                '
      write (unit = 1, fmt = '(a)')     '                                                                            '
      write (unit = 1, fmt = '(a)')     '  A software for calibrating linear climate proxy models                    '
      write (unit = 1, fmt = '(a)')     '                                                                            '
      write (unit = 1, fmt = '(a)')     ' ==========================================================================='
      write (unit = 1, fmt = '(a)')     '                                                                            '
      write (unit = 1, fmt = '(a)')     '  References:                                                               '
      write (unit = 1, fmt = '(a)')     '                                                                            '
      write (unit = 1, fmt = '(a)')     '  Mudelsee M (2023) Unbiased proxy calibration.                             '
      write (unit = 1, fmt = '(a)')     '                    Mathematical Geosciences.                               '
      write (unit = 1, fmt = '(a)')     '                    doi:10.1007/s11004-023-10122-5                          '
      write (unit = 1, fmt = '(a)')     '                    (published online 19 December 2023)                     '
      write (unit = 1, fmt = '(a)')     '                                                                            '
      write (unit = 1, fmt = '(a)')     '  Mudelsee M (2014) Climate Time Series Analysis:                           '
      write (unit = 1, fmt = '(a)')     '                    Classical Statistical and Bootstrap Methods.            '
      write (unit = 1, fmt = '(a)')     '                    Second Edition. Springer, Cham, Switzerland, 454 pp.    '
      write (unit = 1, fmt = '(a)')     '                                                                            '
      write (unit = 1, fmt = '(a)')     '  Copyright (C) Manfred Mudelsee                                            '
      write (unit = 1, fmt = '(a)')     '  https://www.manfredmudelsee.com                                           '
      write (unit = 1, fmt = '(a)')     '                                                                            '
      write (unit = 1, fmt = '(a)')     '  Please see source code (LINCAL.f90) for                                   '
      write (unit = 1, fmt = '(a)')     '  details, disclaimer and permissions.                                      '
      write (unit = 1, fmt = '(a)')     '                                                                            '
      write (unit = 1, fmt = '(a)')     ' ==========================================================================='
!
! 2.  Data and prior knowledge
!     ========================
!
      write (unit = 1, fmt = '(a)')     ' Case    Data                                   Prior knowledge             '
      if (datacase .eq. 1) then
         write (unit = 1, fmt = '(a)')                                                         &
                                        ' Case 1: {      x(i), y(i)}, i = 1, ..., n,     sx (constant), sy (constant)'
      else if (datacase .eq. 2) then
         write (unit = 1, fmt = '(a)')                                                         &
                                        ' Case 2: {t(i), x(i), y(i)}, i = 1, ..., n,     sx (constant), sy (constant)'
      else if (datacase .eq. 3) then
         write (unit = 1, fmt = '(a)')                                                         &
                                        ' Case 3: {      x(i), y(i),                     sx(i), sy(i)}, i = 1, ..., n'
      else if (datacase .eq. 4) then
         write (unit = 1, fmt = '(a)')                                                         &
                                        ' Case 4: {t(i), x(i), y(i),                     sx(i), sy(i)}, i = 1, ..., n'
      end if
      write (unit = 1, fmt = '(a)')             '                                      '
      write (unit = 1, fmt = '(a,a)')           ' Input data file:                     ',trim(adjustl(datafile))
      if (n .lt. 100) then
         write (unit = 1, fmt = '(a, i2)')      ' Sample size:                         ', n
      else if (n .ge. 100 .and. n .lt. 1000) then
         write (unit = 1, fmt = '(a, i3)')      ' Sample size:                         ', n
      else if (n .ge. 1000 .and. n .lt. 10000) then
         write (unit = 1, fmt = '(a, i4)')      ' Sample size:                         ', n
      else if (n .ge. 10000 .and. n .lt. 100000) then
         write (unit = 1, fmt = '(a, i5)')      ' Sample size:                         ', n
      else if (n .ge. 100000) then
         write (unit = 1, fmt = '(a, i6)')      ' Sample size:                         ', n
      end if
      if (datacase .eq. 1 .or. datacase .eq. 2) then
         write (unit = 1, fmt = '(a, f16.8)')   ' sx (constant):                       ', sx_const
         write (unit = 1, fmt = '(a, f16.8)')   ' sy (constant):                       ', sy_const
      else  if (datacase .eq. 3 .or. datacase .eq. 4) then
         write (unit = 1, fmt = '(a, a)')       ' sx(i), sy(i):                        ','see input data file'
      end if
      write (unit = 1, fmt = '(a)')     '                                                                            '
!
! 3.  Estimation result
!     =================
!
      write (unit = 1, fmt = '(a)')                                                             &
         ' ==========================================================================='
      write (unit = 1, fmt = '(a)')                                                             &
         '                                                                            '
      write (unit = 1, fmt = '(a)')                                                             &
         ' Estimation of the linear model Y = beta0 + beta1 * X                       '
      write (unit = 1, fmt = '(a, f5.3)')                                                       &
         ' with bootstrap Student''s t confidence intervals at level ', one - two * alpha
      write (unit = 1, fmt = '(a)')     '                                                                            '
      write (unit = 1, fmt = '(3(a, f16.8), a)')  ' beta0 =  ', beta0, '   [ ', beta0_low, '; ', beta0_upp, ' ]'
      write (unit = 1, fmt = '(3(a, f16.8), a)')  ' beta1 =  ', beta1, '   [ ', beta1_low, '; ', beta1_upp, ' ]'
      write (unit = 1, fmt = '(a)')     '                                                                            '
!
! 4.  Bootstrap parameters
!     ====================
!
      write (unit = 1, fmt = '(a)')                                                             &
         ' ==========================================================================='
      write (unit = 1, fmt = '(a)')                                                             &
         '                                                                            '
      write (unit = 1, fmt = '(a)')                                                             &
         ' Bootstrap (MBBres) resampling parameters                                                                  '
      write (unit = 1, fmt = '(a)')     '                                                                            '
      write (unit = 1, fmt = '(a, i6)')    ' Number of resamples:     ', b
      write (unit = 1, fmt = '(a)')        ' Persistence time estimates (-999.0 means NA)                            '
      write (unit = 1, fmt = '(a, f16.8)') ' tau_x =              ', tau_x
      write (unit = 1, fmt = '(a, f16.8)') ' tau_y =              ', tau_y
      write (unit = 1, fmt = '(a, i6)')    ' Block length:         ', l_mbb
      write (unit = 1, fmt = '(a)')     '                                                                            '
!
! 5.  Prediction result
!     =================
!
      write (unit = 1, fmt = '(a)')                                                             &
         ' ==========================================================================='
      write (unit = 1, fmt = '(a)')                                                             &
         '                                                                            '
      write (unit = 1, fmt = '(a)')                                                             &
         ' Prediction of the linear model Y = beta0 + beta1 * X                       '
      write (unit = 1, fmt = '(a,f5.3)')                                                        &
         ' with bootstrap Student''s t confidence interval at level ', one - two * alpha
      write (unit = 1, fmt = '(a)')                                                             &
         ' taking into account parameter estimation uncertainties                     '
      write (unit = 1, fmt = '(a)')                                                             &
         ' as well as predictor uncertainties assumed of                              '
      write (unit = 1, fmt = '(a)')                                                             &
         ' Gaussian shape with standard deviation equal to                            '
      if (datacase .eq. 1 .or. datacase .eq. 2) then
         write (unit = 1, fmt = '(a,f16.8)')  ' predefined sx (constant)  = ', sx_const
      else  if (datacase .eq. 3 .or. datacase .eq. 4) then
         write (unit = 1, fmt = '(a,f16.8)')  ' assumed constant value sx = ', sx_predict
      end if
      write (unit = 1, fmt = '(a)')                                                             &
         '                                                                            '
      write (unit = 1, fmt = '(6x,3(a,3x),a)')                                                  &
        ' xp             ',                                                                     &
        ' yp             ',                                                                     &
        ' yp_low         ',                                                                     &
        ' yp_upp         '
      do i = 1, np
         write (unit = 1, fmt = '(3(f16.8,3x),f16.8)')                                          &
            xp(i), yp(i), yp_low(i), yp_upp(i)
      end do
!
! 5.  Exit
!     ====
!
      close (unit = 1, status = 'keep')
end subroutine output1
!
! ==================================================================================================
!
function phi_inv(prob)
      use nrtype
      implicit none
      real(sp) :: phi_inv
!
!     Inverse cumulative Gaussian density approximation after
!     Odeh and Evans (1974).
!     
!     The approximation is, for any meaning level of prob,
!     accurate to seven decimal places.
!
!     Reference:
!
!     Odeh RE, Evans JO (1974) The percentage points of the normal distribution.
!                              Applied Statistics 23:96-97.
!
      real(sp), intent(in) :: prob
      real(sp), parameter  :: big  =  1.0e+38_sp
      real(sp), parameter  :: plim =  1.0e-20_sp
      real(sp), parameter  :: p0   = -0.322232431088e0_sp
      real(sp), parameter  :: p1   = -1.0e0_sp
      real(sp), parameter  :: p2   = -0.342242088547e0_sp
      real(sp), parameter  :: p3   = -0.0204231210245e0_sp
      real(sp), parameter  :: p4   = -0.453642210148e-04_sp
      real(sp), parameter  :: q0   =  0.0993484626060e0_sp
      real(sp), parameter  :: q1   =  0.588581570495e0_sp
      real(sp), parameter  :: q2   =  0.531103462366e0_sp
      real(sp), parameter  :: q3   =  0.103537752850e0_sp
      real(sp), parameter  :: q4   =  0.38560700634e-02_sp
      real(sp) ::             p    = -999.0_sp
      real(sp) ::             y    = -999.0_sp
      real(sp), parameter :: zero  = 0.0_sp
      real(sp), parameter :: half  = 0.5_sp
      real(sp), parameter ::  one  = 1.0_sp
      phi_inv = -big
      p = prob
!
!     Ranges for p
!     ============
!
      if (p < plim) then
         phi_inv = -big
      else if (p > one - plim) then
         phi_inv = big
      else if (p == half) then
         phi_inv = zero
      else if (p >= plim .and. p < half) then
         y = sqrt(LOG(1.0_sp / (p * p)))
         phi_inv = y + ((((y*p4 + p3)*y + p2)*y + p1)*y + p0)/((((y*q4 + q3)*y + q2)*y + q1)*y + q0)
         phi_inv = -1.0_sp * phi_inv
      else if (p > half .and. p <= one-plim) then
         p = one-p
         y = sqrt(LOG(1.0_sp / (p * p)))
         phi_inv = y + ((((y*p4 + p3)*y + p2)*y + p1)*y + p0)/((((y*q4 + q3)*y + q2)*y + q1)*y + q0)
      end if
end function phi_inv
!
! ==================================================================================================
!
subroutine prediction1
      use nrtype
      use data1, only: x
      use parameters, only: fp, np
      use result1, only: beta0, beta1, xp, yp
      implicit none
!
!     Predicts yp by means of the estimated linear model.
!     The range, xp, over which predictions are made, is defined as
!     the x-data interval (from xmin to xmax) extended by factor fp
!     to the left and to the right.
!
      integer :: i
      real(sp) :: xmin  = -999.0_sp     ! minimum of x
      real(sp) :: xmax  = -999.0_sp     ! maximum of x
      real(sp) :: xpmin = -999.0_sp     ! minimum of xp
      real(sp) :: xpmax = -999.0_sp     ! maximum of xp
      xmin  = minval(x)
      xmax  = maxval(x)
      xpmin = xmin - fp * (xmax - xmin)
      xpmax = xmax + fp * (xmax - xmin)
      do i = 1, np
         xp(i) = xpmin + real(i - 1) * (xpmax - xpmin) / real(np - 1)
      end do
      call linmod(np, xp, beta0, beta1, yp)
end subroutine prediction1
!
! ==================================================================================================
!
subroutine read1
      use nrtype
      use data1, only: t, x, y
      use prior_knowledge, only: sx, sy
      use setting, only: datacase, datafile, n
      implicit none
!     Reads data.
      character (len = 1) :: flag
      integer :: i
      open (unit = 1, file = datafile, status = 'old',                                          &
            form = 'formatted', action = 'read')
      do while (.true.)
         read (1, '(a1)') flag
         if (flag .ne. '#') then
            backspace (1)
            exit
         end if
      end do
      do i = 1, n
         if (datacase .eq. 1) then
            read (unit = 1, fmt = *)       x(i), y(i)
         else if (datacase .eq. 2) then
            read (unit = 1, fmt = *) t(i), x(i), y(i)
         else if (datacase .eq. 3) then
            read (unit = 1, fmt = *)       x(i), y(i), sx(i), sy(i)
         else if (datacase .eq. 4) then
            read (unit = 1, fmt = *) t(i), x(i), y(i), sx(i), sy(i)
         end if
      end do
      close (unit = 1, status = 'keep')
!
! Test
!
!
!    do i = 1, n
!       if (datacase .eq. 1) then
!          print '(2x,i6,2(2x,f16.3))', i,       x(i), y(i)
!       else if (datacase .eq. 2) then
!          print '(2x,i6,3(2x,f16.3))', i, t(i), x(i), y(i)
!       else if (datacase .eq. 3) then
!          print '(2x,i6,4(2x,f16.3))', i,       x(i), y(i), sx(i), sy(i)
!       else if (datacase .eq. 4) then
!          print '(4x,f8.3,4x,f6.3,4x,f5.2,4x,f5.3,4x,f5.3)', t(i), x(i), y(i), sx(i), sy(i)
!       end if
!    end do
!
end subroutine read1
!
! ==================================================================================================
!
subroutine rhoest(n, x, rho)
      use nrtype
      implicit none
      integer, intent(in) :: n
      real(sp), dimension(n), intent(in) :: x
      real(sp), intent(out) :: rho
!     Estimates autocorrelation coefficient (equidistant data).
!     See also  the book (Mudelsee 2014: Eq. 2.4 therein).
      integer :: i
      real(sp) :: sum1 = -999.0_sp
      real(sp) :: sum2 = -999.0_sp
      sum1 = 0.0_sp
      sum2 = 0.0_sp    
      do i=2, n
         sum1 = sum1 + x(i) * x(i - 1)
         sum2 = sum2 + x(i)**2.0_sp
      end do
      rho = sum1/sum2
end subroutine rhoest
!
! ==================================================================================================
!
subroutine tauest(t_in, x_in, n, tau, rhoout)
      use nrtype
      implicit none
      integer, intent(in) :: n                   ! number of data points
      real(sp), dimension(n), intent(in) :: t_in ! time
      real(sp), dimension(n), intent(in) :: x_in ! time-series values
      real(sp), intent(out) :: tau               ! result: persistence time tau
      real(sp), intent(out) :: rhoout            ! result: equivalent autocorrelation coefficient
!
!     Estimates AR(1) persistence time (tau) and
!     equivalent autocorrelation coefficient (rho),using
!     the least-squares algorithm of Mudelsee (2002).
!
!     Reference:
!
!     Mudelsee M (2002) TAUEST: a computer program for estimating persistence
!                       in unevenly spaced weather/climate time series.
!                       Computers and Geosciences 28:69-72. 
!
!     Notes: (1) assumes t = age (see Point 1)
!
!            (2) automatic bias correction (see Point 5)
!
      real(sp), dimension(n) :: x       ! x renamed
      real(sp), dimension(n) :: t       ! t renamed
      real(sp) :: avex    = -999.0_sp   ! average x
      real(sp) :: varx    = -999.0_sp   ! variance x
      real(sp) :: delta   = -999.0_sp   ! average spacing
      real(sp) :: scalt   = -999.0_sp   !  factor
      real(sp) :: rho     = -999.0_sp   ! rho for x, used in scaling t
      real(sp) :: rho_non = -999.0_sp   ! equivalent autocorrelation coefficient
      real(sp) :: amin    = -999.0_sp   ! output from minls, estimated value of a = exp(-scalt / tau)
      integer ::  mult    = -999        ! flag (multiple solution) mult = 0 if 1 minimum solution,
                                        !                          mult = 1 if more than one minimum
      integer :: i        = -999        ! counter
      real(sp), parameter :: rho_max = 0.99_sp
      real(sp), parameter :: rho_min = 0.01_sp
!
! 1. Rename and change time direction
!    ================================
      do i = n, 1, -1
         t(i) = -1.0_sp *t_in(n + 1 - i)
         x(i) = +1.0_sp *x_in(n + 1 - i)
      end do
!
! 2.  Scaling (x)
!     ===========
!
      call avevar_n(x, n, avex, varx)
      x = x / sqrt(varx)
!
! 3.  Scaling (t)
!     ===========
!
!     => start value of a = 1 / e
!
      delta = abs(t(n) - t(1)) / (n - 1)    ! average sampling
      call rhoest(n, x, rho)                ! rho estimated on x (Mudelsee 2014: Eq. 2.4 therein)
      if (rho <= 0.0_sp) then
         rho = 0.05_sp
      else if (rho >= 1.0_sp) then          ! this setting of rho equal to 0.05 or 0.95 is not
         rho = 0.95_sp                      ! problematic since we require rho only for t-scaling
      end if
      scalt = -1.0_sp*log(rho) / delta      ! scaling factor
      t     = t * scalt                     ! t-scaled: t * (-log(a) / delta)
!
! 4.  Estimation (tau)
!     ================
!
      call minls(n, t, x, amin, mult)
!
! 5.  Result
!     ======
!
      if (mult == 1 .or. amin <= 0.0_sp .or. amin >= 1.0_sp) then
         if (amin <= 0.0_sp) then
            rho_non = rho_min
            tau = -delta / log(rho_min)
         else if (amin >= 1.0_sp) then
            rho_non = rho_max
            tau = -delta / log(rho_max)
         end if
      else
         tau = -1.0_sp / (scalt * log(amin))    ! tau - rescaled, without bias correction tau=-1/log(a)
         rho_non = exp(-delta / tau)            ! equivalent autocorrelation coefficient for tau,
                                                ! used in the bias correction
!
!        Bias correctionfor unknown mean (Kendall, 1954).
!        Equation: a_est = a_est_bc + (1.0_sp + 3.0_sp * a_est_bc) / (n - 1)
!        is solved for a_est_bc.
!
!        Reference:
!
!        Kendall MG (1954) Note on bias in the estimation of autocorrelation.
!                          Biometrika 41:403-404.
!
         rho_non = (rho_non * (n - 1.0_sp) + 1.0_sp) / (n - 4.0_sp)
         if (rho_non >= 1.0_sp) then
            rho_non = exp(-delta / tau) ! if the bias-corrected equivalent autocorrelation coefficient
                                        ! becomes > 1, then the bias correction is not performed
         end if                         ! this may occur if n is small and the autocorrelation large
!
         tau = -delta / log(rho_non)    ! tau, calculated from the bias-corrected rho,
                                        ! a = exp(delta / tau)  gives tau = -delta / log(rho)
      end if
      rhoout = rho_non
end subroutine tauest
!
! ==================================================================================================
!
function t_inv(dof)
      use nrtype
      use setting, only: phi_inv_alpha
      implicit none
      real(sp) :: t_inv
!
!     Inverse Student's t distribution function, approximation
!     after Abramowitz and Stegun (1964).
!
!     Highest inaccuracy occurs when:         o dof is small,
!                                             o alpha is high.
!     For example: dof = 4, alpha = 0.025.
!     There results: t = 2.7756 (instead of 2.776),
!     that is, a negligible error.
!
!     Reference:
!
!     Abramowitz M, Stegun IA (1965) Handbook of Mathematical Functions.
!                                    Dover, New York, NY, 1046 pp.
!
!
      integer,  intent(in) :: dof
      real(sp) :: u = -999.0_sp
      integer  :: d = 0
      u = phi_inv_alpha
      d = dof
      t_inv = u                                                              +                      &
      (1.0_sp /    4) * (                                               u**3 +       u) / d    +    &
      (1.0_sp /   96) * (                             5 * u**5 +   16 * u**3 +   3 * u) / d**2 +    &
      (1.0_sp /  384) * (               3 * u**7 +   19 * u**5 +   17 * u**3 -  15 * u) / d**3 +    &
      (1.0_sp /92160) * ( 79 * u**9 + 776 * u**7 + 1482 * u**5 - 1920 * u**3 - 945 * u) / d**4
end function t_inv
!
! ==================================================================================================
!
subroutine wlsxy(x, y, sigx, sigy, n, a, b)
      use nrtype
      use nrutil, only : assert_eq, swap
      use nr, only : brent
      use chixyfit
      implicit none
      integer, intent(in) :: n                      ! sample size
      real(sp), dimension(n), intent(in) :: x       ! data
      real(sp), dimension(n), intent(in) :: y       ! data
      real(sp), dimension(n), intent(in) :: sigx    ! standard deviation of data x(i) (renamed)
      real(sp), dimension(n), intent(in) :: sigy    ! standard deviation of data x(i) (renamed)
      real(sp), intent(out) :: a                    ! intercept (renamed)
      real(sp), intent(out) :: b                    ! slope (renamed)
!
!     Weighted least-squares for both variables (WLSXY) estimation of a linear regression model.
!
!     References:
!
!     Deming WE (1943)  Statistical Adjustment of Data. Wiley, New York, 261 pp.
!
!     Lindley DV (1947) Regression lines and the linear functional relationship.
!                       Journal of the Royal Statistical Society, Supplement 9:218-244.
!
!     Mudelsee M (2014) Climate Time Series Analysis:                       
!                       Classical Statistical and Bootstrap Methods.        
!                       Second Edition. Springer, Cham, Switzerland, 454 pp.
!                       [Chapter 8 therein]
!
!     Press WH, Teukolsky SA, Vetterling WT, Flannery BP (1992b) Numerical Recipes in Fortran 77.
!                       Second Edition. Cambridge University Press, Cambridge, United Kingdom, 933 pp.
!
!     Reed BC (1989)    Linear least-squares fits with errors in both coordinates.
!                       American Journal of Physics 57:642-646.
!                       [Corrigendum: American Journal of Physics, Vol. 58, No. 2 (February 1990), p. 189]
!
!     Reed BC (1992)    Linear least-squares fits with errors in both coordinates.
!                       II: Comments on parameter variances.
!                       American Journal of Physics 60:59-62.
!
!     Squire PT (1990)  Comment on "Linear least-squares fits with errors in both coordinates,"
!                       by B. C. Reed [Am. J. Phys. 57, 642-646 (1989)].
!                       American Journal of Physics 58:1209.
!
!     York D (1966)     Least-squares fitting of a straight line.
!                       Canadian Journal of Physics 44:1079-1086.
!
      real(sp), parameter :: potn = 1.571000_sp
      real(sp), parameter :: big  = 1.0e30_sp
      real(sp), parameter :: acc  = 1.0e-3_sp
      integer(i4b) :: j
      real(sp), dimension(n), target :: xx
      real(sp), dimension(n), target :: yy
      real(sp), dimension(n), target :: sx
      real(sp), dimension(n), target :: sy
      real(sp), dimension(n), target :: ww
      real(sp), dimension(6) :: ang  = -999.0_sp
      real(sp), dimension(6) :: ch   = -999.0_sp
      real(sp) :: amx   = -999.0_sp
      real(sp) :: amn   = -999.0_sp
      real(sp) :: varx  = -999.0_sp
      real(sp) :: vary  = -999.0_sp
      real(sp) :: scale = -999.0_sp
      real(sp) :: bmn   = -999.0_sp
      real(sp) :: bmx   = -999.0_sp
      real(sp) :: d1    = -999.0_sp
      real(sp) :: d2    = -999.0_sp
      real(sp) :: r2    = -999.0_sp
      real(sp) :: chi2  = -999.0_sp
      real(sp) :: dum1  = -999.0_sp
      real(sp) :: dum2  = -999.0_sp
      real(sp) :: dum3  = -999.0_sp
      real(sp) :: dum4  = -999.0_sp
      real(sp) :: dum5  = -999.0_sp
      xxp => xx
      yyp => yy
      sxp => sx
      syp => sy
      wwp => ww
      call avevar_n(x, n, dum1, varx)
      call avevar_n(y, n, dum1, vary)
      scale = sqrt(varx / vary)
      xx(:) = x(:)
      yy(:) = y(:) * scale
      sx(:) = sigx(:)
      sy(:) = sigy(:) * scale
      ww(:) = sqrt(sx(:)**2 + sy(:)**2)
      call ols(xx, yy, n, dum1, b)
      offs   = 0.0_sp
      ang(1) = 0.0_sp
      ang(2) = atan(b)
      ang(4) = 0.0_sp
      ang(5) = ang(2)
      ang(6) = potn
      do j = 4, 6
         ch(j) = chixy(ang(j))
      end do
      call mmbrak(ang(1), ang(2), ang(3), ch(1), ch(2), ch(3), chixy)
      chi2 = brent(ang(1), ang(2), ang(3), chixy, acc, b)
      chi2 = chixy(b)
      a = aa
      a = a / scale
      b = tan(b) / scale
end subroutine wlsxy
!
! ==================================================================================================
!
